import 'package:flutter/material.dart';
import 'package:flutter_bloc/flutter_bloc.dart';
import '../../../../../../injection/injection.dart';
import '../../bloc/recommendations/p2p_recommendations_bloc.dart';
import '../../../../../../core/widgets/error_widget.dart' as core_widgets;
import '../../../../../../core/theme/global_theme_extensions.dart';

/// Smart P2P Recommendations Widget
///
/// Displays personalized trading recommendations based on user activity
/// and market analysis with proper theming and animations
class SmartRecommendationsWidget extends StatelessWidget {
  const SmartRecommendationsWidget({super.key});

  @override
  Widget build(BuildContext context) {
    return BlocProvider(
      create: (context) => getIt<P2PRecommendationsBloc>()
        ..add(const P2PRecommendationsLoadRequested()),
      child: const _SmartRecommendationsContent(),
    );
  }
}

class _SmartRecommendationsContent extends StatelessWidget {
  const _SmartRecommendationsContent();

  @override
  Widget build(BuildContext context) {
    return BlocBuilder<P2PRecommendationsBloc, P2PRecommendationsState>(
      builder: (context, state) {
        if (state is P2PRecommendationsLoading) {
          return const _LoadingRecommendations();
        }

        if (state is P2PRecommendationsError) {
          return core_widgets.ErrorWidget(
            message: state.failure.message,
            onRetry: () {
              context
                  .read<P2PRecommendationsBloc>()
                  .add(const P2PRecommendationsLoadRequested());
            },
          );
        }

        // For any loaded state, show our mock recommendations for now
        // This provides a working UI while the real data integration is completed
        if (state is P2PRecommendationsLoaded ||
            state is P2PRecommendationsOfferSuggestionsLoaded ||
            state is P2PRecommendationsPriceAlertsLoaded) {
          final recommendations = _createMockRecommendations();

          if (recommendations.isEmpty) {
            return const _EmptyRecommendations();
          }

          return Container(
            decoration: BoxDecoration(
              gradient: LinearGradient(
                colors: [
                  context.cardBackground,
                  context.cardBackground.withValues(alpha: 0.8),
                ],
                begin: Alignment.topCenter,
                end: Alignment.bottomCenter,
              ),
              borderRadius: BorderRadius.circular(16),
            ),
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                _buildHeader(context),
                const SizedBox(height: 16),
                _buildRecommendationsList(context, recommendations),
                const SizedBox(height: 12),
                _buildFooter(context, recommendations.length),
              ],
            ),
          );
        }

        return const SizedBox.shrink();
      },
    );
  }

  Widget _buildHeader(BuildContext context) {
    return Padding(
      padding: const EdgeInsets.all(20),
      child: Row(
        children: [
          Container(
            padding: const EdgeInsets.all(12),
            decoration: BoxDecoration(
              color: context.colors.primary.withValues(alpha: 0.1),
              borderRadius: BorderRadius.circular(12),
            ),
            child: Icon(
              Icons.lightbulb_outline,
              color: context.colors.primary,
              size: 24,
            ),
          ),
          const SizedBox(width: 16),
          Expanded(
            child: Column(
              crossAxisAlignment: CrossAxisAlignment.start,
              children: [
                Text(
                  'Smart Recommendations',
                  style: context.h5.copyWith(
                    fontWeight: FontWeight.bold,
                    color: context.textPrimary,
                  ),
                ),
                const SizedBox(height: 4),
                Text(
                  'Personalized trading suggestions',
                  style: context.bodyM.copyWith(
                    color: context.textSecondary,
                  ),
                ),
              ],
            ),
          ),
          IconButton(
            onPressed: () {
              context
                  .read<P2PRecommendationsBloc>()
                  .add(const P2PRecommendationsLoadRequested());
            },
            icon: Icon(
              Icons.refresh,
              color: context.textSecondary,
            ),
          ),
        ],
      ),
    );
  }

  Widget _buildRecommendationsList(
      BuildContext context, List<MockRecommendation> recommendations) {
    return Padding(
      padding: const EdgeInsets.symmetric(horizontal: 20),
      child: Column(
        children: recommendations
            .take(3)
            .map((recommendation) =>
                _buildRecommendationCard(context, recommendation))
            .toList(),
      ),
    );
  }

  Widget _buildRecommendationCard(
      BuildContext context, MockRecommendation recommendation) {
    return Container(
      margin: const EdgeInsets.only(bottom: 12),
      padding: const EdgeInsets.all(16),
      decoration: BoxDecoration(
        color: context.cardBackground,
        borderRadius: BorderRadius.circular(12),
        border: Border.all(
          color: context.borderColor,
          width: 1,
        ),
      ),
      child: Column(
        crossAxisAlignment: CrossAxisAlignment.start,
        children: [
          Row(
            children: [
              Container(
                padding: const EdgeInsets.all(8),
                decoration: BoxDecoration(
                  color: _getRecommendationColor(context, recommendation.type)
                      .withValues(alpha: 0.1),
                  borderRadius: BorderRadius.circular(8),
                ),
                child: Icon(
                  _getRecommendationIcon(recommendation.type),
                  color: _getRecommendationColor(context, recommendation.type),
                  size: 20,
                ),
              ),
              const SizedBox(width: 12),
              Expanded(
                child: Column(
                  crossAxisAlignment: CrossAxisAlignment.start,
                  children: [
                    Text(
                      recommendation.title,
                      style: context.bodyL.copyWith(
                        color: context.textPrimary,
                        fontWeight: FontWeight.w600,
                      ),
                    ),
                    const SizedBox(height: 2),
                    Text(
                      recommendation.type,
                      style: context.bodyS.copyWith(
                        color: _getRecommendationColor(
                            context, recommendation.type),
                        fontWeight: FontWeight.w500,
                      ),
                    ),
                  ],
                ),
              ),
              Container(
                padding: const EdgeInsets.symmetric(horizontal: 8, vertical: 4),
                decoration: BoxDecoration(
                  color: _getPriorityColor(context, recommendation.priority)
                      .withValues(alpha: 0.1),
                  borderRadius: BorderRadius.circular(12),
                ),
                child: Text(
                  _getPriorityLabel(recommendation.priority),
                  style: context.bodyS.copyWith(
                    color: _getPriorityColor(context, recommendation.priority),
                    fontWeight: FontWeight.w600,
                  ),
                ),
              ),
            ],
          ),
          const SizedBox(height: 12),
          Text(
            recommendation.description,
            style: context.bodyM.copyWith(
              color: context.textSecondary,
            ),
          ),
          const SizedBox(height: 12),
          Row(
            children: [
              if (recommendation.potentialSavings != null) ...[
                Icon(
                  Icons.savings_outlined,
                  size: 16,
                  color: context.buyColor,
                ),
                const SizedBox(width: 4),
                Text(
                  'Save ${recommendation.potentialSavings}',
                  style: context.bodyS.copyWith(
                    color: context.buyColor,
                    fontWeight: FontWeight.w500,
                  ),
                ),
                const SizedBox(width: 16),
              ],
              Icon(
                Icons.schedule,
                size: 16,
                color: context.textTertiary,
              ),
              const SizedBox(width: 4),
              Text(
                recommendation.timeEstimate,
                style: context.bodyS.copyWith(
                  color: context.textTertiary,
                ),
              ),
            ],
          ),
        ],
      ),
    );
  }

  Widget _buildFooter(BuildContext context, int recommendationCount) {
    return Padding(
      padding: const EdgeInsets.symmetric(horizontal: 20),
      child: Container(
        padding: const EdgeInsets.all(16),
        decoration: BoxDecoration(
          color: context.colors.primary.withValues(alpha: 0.05),
          borderRadius: BorderRadius.circular(12),
          border: Border.all(
            color: context.colors.primary.withValues(alpha: 0.1),
            width: 1,
          ),
        ),
        child: Row(
          children: [
            Icon(
              Icons.info_outline,
              color: context.colors.primary,
              size: 20,
            ),
            const SizedBox(width: 12),
            Expanded(
              child: Text(
                'Showing $recommendationCount personalized recommendations based on your trading history',
                style: context.bodyS.copyWith(
                  color: context.textSecondary,
                ),
              ),
            ),
          ],
        ),
      ),
    );
  }

  // Create mock recommendations for demonstration
  List<MockRecommendation> _createMockRecommendations() {
    return [
      const MockRecommendation(
        title: 'Great BTC Buy Opportunity',
        type: 'Buy Opportunity',
        description:
            'Found a competitive BTC offer 2% below market price with your preferred payment method.',
        priority: 'high',
        potentialSavings: '\$45',
      ),
      const MockRecommendation(
        title: 'New Payment Method Available',
        type: 'Payment Method',
        description:
            'Bank Transfer now available in your region with faster settlement times.',
        priority: 'medium',
        timeEstimate: '2-3 min',
      ),
      const MockRecommendation(
        title: 'ETH Price Alert Triggered',
        type: 'Price Alert',
        description:
            'ETH has reached your target price of \$2,400. Multiple offers available.',
        priority: 'high',
        potentialSavings: '\$25',
      ),
    ];
  }

  Color _getRecommendationColor(BuildContext context, String type) {
    switch (type.toLowerCase()) {
      case 'buy opportunity':
        return context.buyColor;
      case 'sell opportunity':
        return context.sellColor;
      case 'price alert':
        return context.warningColor;
      case 'market trend':
        return context.colors.tertiary;
      case 'payment method':
        return context.colors.secondary;
      default:
        return context.colors.primary;
    }
  }

  IconData _getRecommendationIcon(String type) {
    switch (type.toLowerCase()) {
      case 'buy opportunity':
        return Icons.trending_up;
      case 'sell opportunity':
        return Icons.trending_down;
      case 'price alert':
        return Icons.notifications_active;
      case 'market trend':
        return Icons.analytics;
      case 'payment method':
        return Icons.payment;
      default:
        return Icons.lightbulb_outline;
    }
  }

  Color _getPriorityColor(BuildContext context, String priority) {
    switch (priority.toLowerCase()) {
      case 'high':
        return context.colors.error;
      case 'medium':
        return context.warningColor;
      case 'low':
        return context.colors.secondary;
      default:
        return context.colors.tertiary;
    }
  }

  String _getPriorityLabel(String priority) {
    switch (priority.toLowerCase()) {
      case 'high':
        return 'High Priority';
      case 'medium':
        return 'Medium';
      case 'low':
        return 'Low';
      default:
        return priority;
    }
  }
}

class _LoadingRecommendations extends StatelessWidget {
  const _LoadingRecommendations();

  @override
  Widget build(BuildContext context) {
    return Container(
      margin: const EdgeInsets.symmetric(horizontal: 20),
      padding: const EdgeInsets.all(20),
      decoration: BoxDecoration(
        color: context.cardBackground,
        borderRadius: BorderRadius.circular(16),
        border: Border.all(
          color: context.borderColor,
          width: 1,
        ),
      ),
      child: Center(
        child: Column(
          children: [
            CircularProgressIndicator(
              valueColor: AlwaysStoppedAnimation<Color>(context.colors.primary),
            ),
            const SizedBox(height: 12),
            Text(
              'Analyzing your trading patterns...',
              style: context.bodyM.copyWith(
                color: context.textSecondary,
              ),
            ),
          ],
        ),
      ),
    );
  }
}

class _EmptyRecommendations extends StatelessWidget {
  const _EmptyRecommendations();

  @override
  Widget build(BuildContext context) {
    return Container(
      margin: const EdgeInsets.symmetric(horizontal: 20),
      padding: const EdgeInsets.all(32),
      decoration: BoxDecoration(
        color: context.cardBackground,
        borderRadius: BorderRadius.circular(16),
        border: Border.all(
          color: context.borderColor,
          width: 1,
        ),
      ),
      child: Center(
        child: Column(
          children: [
            Container(
              padding: const EdgeInsets.all(16),
              decoration: BoxDecoration(
                color: context.colors.primary.withValues(alpha: 0.1),
                borderRadius: BorderRadius.circular(50),
              ),
              child: Icon(
                Icons.lightbulb_outline,
                color: context.colors.primary,
                size: 32,
              ),
            ),
            const SizedBox(height: 16),
            Text(
              'No Recommendations Yet',
              style: context.h6.copyWith(
                color: context.textPrimary,
                fontWeight: FontWeight.w600,
              ),
            ),
            const SizedBox(height: 8),
            Text(
              'Complete a few trades to receive personalized recommendations',
              style: context.bodyM.copyWith(
                color: context.textSecondary,
              ),
              textAlign: TextAlign.center,
            ),
          ],
        ),
      ),
    );
  }
}

// Mock recommendation class for demonstration
class MockRecommendation {
  const MockRecommendation({
    required this.title,
    required this.type,
    required this.description,
    required this.priority,
    this.potentialSavings,
    this.timeEstimate = '5-10 min',
  });

  final String title;
  final String type;
  final String description;
  final String priority;
  final String? potentialSavings;
  final String timeEstimate;
}
